// This file has been prepared for Doxygen automatic documentation generation.
/** \file ********************************************************************
*
* 
*
* - File              : uart.h
* - Compiler          : WinAVR 20100110
* - Supported devices : ATMega8
* - Author			  : Dominic Rathje (dominic.rathje@uni-ulm.de)
*
* - Description       : Implementation of USART Stack
*
* $Revision: 1.0 $
*****************************************************************************/

#ifndef _UART_H_
#define _UART_H_ 1



#define UART_DDR	DDRD
#define RXPIN_bm	(1<<2)
#define TXPIN_bm	(1<<3)

#define UART_DATA	UDR

#define RXC_vect	USART_RXC_vect
#define DRE_vect	USART_UDRE_vect

//Buffer size can be 2,4,8,16,32,64,128,256
#define UART_RX_BUFFER_SIZE		64
#define UART_TX_BUFFER_SIZE 	64

#define UBRR_val 	25 		//38400
//#define UBRR_val 	8 		//115200

#define UART_PRINTF_COMPATIBILITY 1

/** \brief Initialise the USART Module
 * 
 *  \param void
 *  \return void
 */
void uart_init(void);

/** \brief Send one Byte
 * 
 * This fuction sends one byte by copying the data into the fifo-buffer
 * and enabling the UDRE-Interrupt. The actual transmission is then 
 * initiated by the Interrupt-Service-Routine. If the buffer is full
 * this function waits until the next byte has been transmitted, so it
 * is always successful.
 * 
 * \sa uart_putchar for the printf compatible version
 * \param data Data-Byte to send
 * \return void
 */
void uart_sendbyte(uint8_t data);

/** \brief Pull one Byte from RX-Buffer
 * 
 * If the RX-Fifo is enabled any data receives is automaticaly pusched 
 * into the Fifo ba the ISR. This function removes the oldest Byte from the Buffer
 * and copies it into the Memory-Space which is handed over as a parameter.
 * 
 * \sa uart_getchar for the scanf compatible version
 * \param data pointer to a buffer where the received data is copied to
 * \return $true$ if data has been copied. $false$ if the RX buffer was empty
 */
uint8_t uart_getbyte(uint8_t *data);

/** \brief Send one data byte without buffering in the FIFO.
 * 
 * \param data Data-Byte to send
 * \return true if data has been send. else false
 */
uint8_t uart_senddata(uint8_t data);

/** \brief Send one id byte without buffering in the FIFO.
 * 
 * This function is only for use in 9-Bit Character Size and MPCM Mode.
 * The ninth bit es set to 1 to send an address/id frame.
 * 
 * \param data Data-Byte to send
 * \return true if data has been send. else false
 */
uint8_t uart_sendid(uint8_t data);


/** \brief Enable Multiprocessor Communication Mode.
 * 
 * \param void
 * \return void
 */
inline void uart_mpcm_en(void) {
	UCSRA |= (1<<MPCM);
}

/** \brief Disable Multiprocessor Communication Mode.
 * 
 * \param void
 * \return void
 */
inline void uart_mpcm_dis(void) {
	UCSRA &= ~(1<<MPCM);
}

/** \brief Enable Data Register Empty Interrupt.
 * 
 * \param void
 * \return void
 */
inline void uart_udre_int_en(void) {
	UCSRB |= (1<<UDRIE);
}

/** \brief Disable Data Register Empty Interrupt.
 * 
 * \param void
 * \return void
 */
inline void uart_udre_int_dis(void) {
	UCSRB &= ~(1<<UDRIE);
}

/** \brief Enable Data RX Complete Interrupt.
 * 
 * \param void
 * \return void
 */
inline void uart_rxc_int_en(void) {
	UCSRB |= (1<<RXCIE);
}

/** \brief Disable Data RX Complete Interrupt.
 * 
 * \param void
 * \return void
 */
inline void uart_rxc_int_dis(void) {
	UCSRB &= ~(1<<RXCIE);
}


#ifdef UART_PRINTF_COMPATIBILITY
#include <stdio.h>

/**  \brief Send one Byte (printf version)
 * 
 * This fuction sends one byte by copyint the data into the fifo-buffer
 * and enabling the UDRE-Interrupt. The actual transmission is then 
 * initiated by the Interrupt-Service-Routine. If the buffer is full
 * this function waits until the next byte has been transmitted, so it
 * is always successful.
 * 
 * This is the printf compatible version ov uart_sendbyte
 * 
 * \sa uart_sendbyte for the standars version
 * \param data Data-Byte to send
 * \return void
 */
int uart_putchar(char c, FILE *sream);

/**  \brief 
 * 
 * \param 
 * \return
 */
int uart_getchar(FILE *stream);

#endif

#endif //_UART_H_
