// This file has been prepared for Doxygen automatic documentation generation.
/** \file ********************************************************************
*
* 
*
* - File              : uart1.h
* - Compiler          : WinAVR 20100110
* - Supported devices : ATMXmega128A1
* - Author			  : Dominic Rathje (dominic.rathje@uni-ulm.de)
*
* - Description       : Implementation of USART Stack
*
* $Revision: 1.0 $
*****************************************************************************/

#include <avr/io.h>
#include <stdio.h>

#ifndef _UART1_H_
#define _UART1_H_ 1

#define UART1_PORT				PORTC
#define UART1_RXPIN_bm			PIN6_bm
#define UART1_TXPIN_bm			PIN7_bm

#define UART1					USARTC1
#define UART1_RXC_vect			USARTC1_RXC_vect
#define UART1_DRE_vect			USARTC1_DRE_vect

//Buffer size can be 2,4,8,16,32,64,128,256
#define UART1_RX_BUFFER_SIZE	64
#define UART1_TX_BUFFER_SIZE 	64

#define UART1_BSCALE		-4
// BSEL = 1/(2^(BSCALE))*(F_CPU/(16*BAUD)-1)
//#define UART1_BSEL		262		// 115200
#define UART1_BSEL		817		// 38400

#define UART1_CHSIZE_gc			USART_CHSIZE_9BIT_gc

/** \brief Initialise the USART Module
 * 
 *  \param void
 *  \return void
 */
void uart1_init(void);

/** \brief Send one Byte
 * 
 * This fuction sends one byte by copying the data into the fifo-buffer
 * and enabling the UDRE-Interrupt. The actual transmission is then 
 * initiated by the Interrupt-Service-Routine. If the buffer is full
 * this function waits until the next byte has been transmitted, so it
 * is always successful.
 * 
 * \sa uart1_putchar for the printf compatible version
 * \param data Data-Byte to send
 * \return void
 */
void uart1_sendbyte(uint8_t data);

/** \brief Pull one Byte from RX-Buffer
 * 
 * If the RX-Fifo is enabled any data receives is automaticaly pusched 
 * into the Fifo ba the ISR. This function removes the oldest Byte from the Buffer
 * and copies it into the Memory-Space which is handed over as a parameter.
 * 
 * \sa uart1_getchar for the scanf compatible version
 * \param data pointer to a buffer where the received data is copied to
 * \return $true$ if data has been copied. $false$ if the RX buffer was empty
 */
uint8_t uart1_getbyte(uint8_t *data);

/** \brief Send one data byte without buffering in the FIFO.
 * 
 * \param data Data-Byte to send
 * \return true if data has been send. else false
 */
uint8_t uart1_senddata(uint8_t data);

/** \brief Send one id byte without buffering in the FIFO.
 * 
 * This function is only for use in 9-Bit Character Size and MPCM Mode.
 * The ninth bit es set to 1 to send an address/id frame.
 * 
 * \param data Data-Byte to send
 * \return true if data has been send. else false
 */
uint8_t uart1_sendid(uint8_t data);

/**  \brief Send one Byte (printf version)
 * 
 * This fuction sends one byte by copyint the data into the fifo-buffer
 * and enabling the UDRE-Interrupt. The actual transmission is then 
 * initiated by the Interrupt-Service-Routine. If the buffer is full
 * this function waits until the next byte has been transmitted, so it
 * is always successful.
 * 
 * This is the printf compatible version ov uart1_sendbyte
 * 
 * \sa uart1_sendbyte for the standars version
 * \param data Data-Byte to send
 * \return void
 */
int uart1_putchar(char c, FILE *sream);

/** \brief Enable Multiprocessor Communication Mode.
 * 
 * \param void
 * \return void
 */
inline void uart1_mpcm_en(void) {
	UART1.CTRLB |= USART_MPCM_bm;
}

/** \brief Disable Multiprocessor Communication Mode.
 * 
 * \param void
 * \return void
 */
inline void uart1_mpcm_dis(void) {
	UART1.CTRLB &= ~USART_MPCM_bm;
}

/** \brief Enable Data Register Empty Interrupt.
 * 
 * \param void
 * \return void
 */
inline void uart1_udre_int_en(void) {
	UART1.CTRLA |= USART_DREINTLVL_LO_gc;
}

/** \brief Disable Data Register Empty Interrupt.
 * 
 * \param void
 * \return void
 */
inline void uart1_udre_int_dis(void) {
	UART1.CTRLA &= ~USART_DREINTLVL_LO_gc;
}

/** \brief Enable Data RX Complete Interrupt.
 * 
 * \param void
 * \return void
 */
inline void uart1_rxc_int_en(void) {
	UART1.CTRLA |= USART_RXCINTLVL_LO_gc;
}

/** \brief Disable Data RX Complete Interrupt.
 * 
 * \param void
 * \return void
 */
inline void uart1_rxc_int_dis(void) {
	UART1.CTRLA &= ~USART_RXCINTLVL_LO_gc;
}

#endif //_UART1_H_
